<?php
/**
 * Plugin Name: CODLAB Booking Widget
 * Plugin URI: https://www.codlab.de
 * Description: Widget popup per prenotazioni online - Configurabile e multilingua
 * Version: 2.2.0
 * Author: CODLAB
 * Author URI: https://www.codlab.de
 * License: GPL v2 or later
 * Text Domain: codlab-booking-widget
 */

if (!defined('ABSPATH')) {
    exit;
}

class CODLAB_Booking_Widget {

    private static $instance = null;
    private $options;
    private $translations;
    private $current_lang;

    private $lang_strings = array(
        'en' => array('button_text' => 'Book Now', 'header_title' => 'Online Reservation', 'close' => 'Close'),
        'de' => array('button_text' => 'Jetzt reservieren', 'header_title' => 'Online Reservierung', 'close' => 'Schließen'),
        'it' => array('button_text' => 'Prenota Ora', 'header_title' => 'Prenotazione Online', 'close' => 'Chiudi'),
        'fr' => array('button_text' => 'Réserver', 'header_title' => 'Réservation en ligne', 'close' => 'Fermer'),
        'es' => array('button_text' => 'Reservar Ahora', 'header_title' => 'Reserva Online', 'close' => 'Cerrar'),
        'pt' => array('button_text' => 'Reservar Agora', 'header_title' => 'Reserva Online', 'close' => 'Fechar'),
        'nl' => array('button_text' => 'Nu reserveren', 'header_title' => 'Online Reserveren', 'close' => 'Sluiten'),
        'pl' => array('button_text' => 'Zarezerwuj', 'header_title' => 'Rezerwacja Online', 'close' => 'Zamknij'),
        'ru' => array('button_text' => 'Забронировать', 'header_title' => 'Онлайн бронирование', 'close' => 'Закрыть'),
        'zh' => array('button_text' => '立即预订', 'header_title' => '在线预订', 'close' => '关闭'),
        'ja' => array('button_text' => '今すぐ予約', 'header_title' => 'オンライン予約', 'close' => '閉じる'),
        'ko' => array('button_text' => '지금 예약', 'header_title' => '온라인 예약', 'close' => '닫기'),
        'tr' => array('button_text' => 'Şimdi Rezerve Et', 'header_title' => 'Online Rezervasyon', 'close' => 'Kapat'),
        'ar' => array('button_text' => 'احجز الآن', 'header_title' => 'الحجز عبر الإنترنت', 'close' => 'إغلاق'),
        'cs' => array('button_text' => 'Rezervovat', 'header_title' => 'Online Rezervace', 'close' => 'Zavřít'),
        'da' => array('button_text' => 'Book nu', 'header_title' => 'Online Reservation', 'close' => 'Luk'),
        'sv' => array('button_text' => 'Boka nu', 'header_title' => 'Online Bokning', 'close' => 'Stäng'),
        'no' => array('button_text' => 'Bestill nå', 'header_title' => 'Online Reservasjon', 'close' => 'Lukk'),
        'fi' => array('button_text' => 'Varaa nyt', 'header_title' => 'Online-varaus', 'close' => 'Sulje'),
        'el' => array('button_text' => 'Κράτηση τώρα', 'header_title' => 'Online Κράτηση', 'close' => 'Κλείσιμο'),
        'hu' => array('button_text' => 'Foglalás most', 'header_title' => 'Online Foglalás', 'close' => 'Bezárás'),
        'ro' => array('button_text' => 'Rezervă acum', 'header_title' => 'Rezervare Online', 'close' => 'Închide'),
        'uk' => array('button_text' => 'Забронювати', 'header_title' => 'Онлайн бронювання', 'close' => 'Закрити'),
        'hr' => array('button_text' => 'Rezerviraj sada', 'header_title' => 'Online Rezervacija', 'close' => 'Zatvori'),
        'sk' => array('button_text' => 'Rezervovať', 'header_title' => 'Online Rezervácia', 'close' => 'Zavrieť'),
        'sl' => array('button_text' => 'Rezerviraj zdaj', 'header_title' => 'Spletna rezervacija', 'close' => 'Zapri'),
    );

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->current_lang = $this->detect_language();
        $this->translations = $this->get_language_strings($this->current_lang);
        
        $defaults = array(
            'booking_url'       => '',
            'button_text'       => '',
            'header_title'      => '',
            'primary_color'     => '#F58220',
            'text_color'        => '#ffffff',
            'position'          => 'bottom-right',
            'button_style'      => 'rounded',
            'show_on_mobile'    => '1',
            'popup_width'       => '420',
            'popup_height'      => '700',
            'z_index'           => '999999',
            'enabled'           => '1',
            'auto_language'     => '1',
            'show_header'       => '0', // Default OFF per evitare ridondanze
        );
        
        $this->options = wp_parse_args(get_option('codlab_booking_options', array()), $defaults);
        
        add_action('wp_footer', array($this, 'render_widget'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));
    }

    private function detect_language() {
        if (defined('ICL_LANGUAGE_CODE')) return ICL_LANGUAGE_CODE;
        if (function_exists('pll_current_language')) {
            $lang = pll_current_language('slug');
            if ($lang) return $lang;
        }
        if (class_exists('TRP_Translate_Press')) {
            global $TRP_LANGUAGE;
            if (!empty($TRP_LANGUAGE)) return substr($TRP_LANGUAGE, 0, 2);
        }
        if (function_exists('weglot_get_current_language')) return weglot_get_current_language();
        if (function_exists('qtranxf_getLanguage')) return qtranxf_getLanguage();
        return substr(get_locale(), 0, 2);
    }

    private function get_language_strings($lang) {
        return $this->lang_strings[$lang] ?? $this->lang_strings['en'];
    }

    private function get_text($key) {
        $option_value = $this->options[$key] ?? '';
        if (!empty($option_value)) return $option_value;
        if ($this->options['auto_language'] === '1' && isset($this->translations[$key])) {
            return $this->translations[$key];
        }
        return array('button_text' => 'Book Now', 'header_title' => 'Online Reservation', 'close' => 'Close')[$key] ?? '';
    }

    public function render_widget() {
        if ($this->options['enabled'] !== '1' || empty($this->options['booking_url'])) return;
        
        if ($this->options['show_on_mobile'] !== '1') {
            echo '<style>@media(max-width:768px){#codlab-booking-widget{display:none!important;}}</style>';
        }

        $this->render_styles();
        $this->render_html();
        $this->render_scripts();
    }

    private function render_styles() {
        $primary = esc_attr($this->options['primary_color']);
        $text = esc_attr($this->options['text_color']);
        $width = intval($this->options['popup_width']);
        $height = intval($this->options['popup_height']);
        $zindex = intval($this->options['z_index']);
        $button_style = $this->options['button_style'];
        $show_header = $this->options['show_header'] === '1';

        $pos_css = $this->get_position_css($this->options['position']);
        $popup_pos_css = $this->get_popup_position_css($this->options['position']);

        $btn_radius = $button_style === 'rounded' ? '50px' : ($button_style === 'circle' ? '50%' : '8px');
        $btn_padding = $button_style === 'circle' ? '16px' : '14px 24px';
        ?>
        <style id="codlab-booking-styles">
        #codlab-booking-widget {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            position: fixed;
            <?php echo $pos_css; ?>
            z-index: <?php echo $zindex; ?>;
        }
        #codlab-chat-btn {
            display: inline-flex;
            align-items: center;
            gap: 10px;
            padding: <?php echo $btn_padding; ?>;
            background: <?php echo $primary; ?>;
            color: <?php echo $text; ?>;
            border: none;
            border-radius: <?php echo $btn_radius; ?>;
            cursor: pointer;
            font-size: 16px;
            font-weight: 600;
            box-shadow: 0 4px 24px rgba(0,0,0,0.25);
            transition: all 0.3s ease;
        }
        #codlab-chat-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 30px rgba(0,0,0,0.35);
        }
        #codlab-chat-btn svg { width: 22px; height: 22px; }
        #codlab-chat-btn.active .btn-text,
        #codlab-chat-btn.active .icon-open { display: none; }
        #codlab-chat-btn .icon-close { display: none; }
        #codlab-chat-btn.active .icon-close { display: block; }
        #codlab-chat-btn.active { padding: 14px; border-radius: 50%; }
        
        #codlab-popup {
            position: fixed;
            <?php echo $popup_pos_css; ?>
            width: <?php echo $width; ?>px;
            height: <?php echo $height; ?>px;
            max-width: calc(100vw - 30px);
            max-height: calc(100vh - 100px);
            background: #fff;
            border-radius: 16px;
            box-shadow: 0 12px 48px rgba(0,0,0,0.25);
            overflow: hidden;
            display: none;
            flex-direction: column;
            z-index: <?php echo $zindex + 1; ?>;
        }
        #codlab-popup.open {
            display: flex;
            animation: codlab-fadeIn 0.3s ease;
        }
        @keyframes codlab-fadeIn {
            from { opacity: 0; transform: scale(0.95) translateY(10px); }
            to { opacity: 1; transform: scale(1) translateY(0); }
        }
        
        <?php if ($show_header): ?>
        .codlab-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 12px 16px;
            background: <?php echo $primary; ?>;
            color: <?php echo $text; ?>;
            flex-shrink: 0;
        }
        .codlab-header-title { font-size: 16px; font-weight: 600; }
        .codlab-close {
            width: 30px; height: 30px;
            display: flex; align-items: center; justify-content: center;
            background: rgba(255,255,255,0.2);
            border: none; border-radius: 50%;
            cursor: pointer; color: <?php echo $text; ?>;
            transition: all 0.2s;
        }
        .codlab-close:hover { background: rgba(255,255,255,0.3); transform: rotate(90deg); }
        .codlab-close svg { width: 16px; height: 16px; }
        <?php endif; ?>
        
        .codlab-body {
            flex: 1;
            position: relative;
            overflow: hidden;
            border-radius: <?php echo $show_header ? '0' : '16px'; ?>;
        }
        .codlab-body iframe {
            width: 100%;
            height: 100%;
            border: none;
            display: block;
        }
        .codlab-body.loading::after {
            content: '';
            position: absolute;
            top: 50%; left: 50%;
            width: 36px; height: 36px;
            margin: -18px 0 0 -18px;
            border: 3px solid rgba(245,130,32,0.2);
            border-top-color: <?php echo $primary; ?>;
            border-radius: 50%;
            animation: codlab-spin 0.8s linear infinite;
        }
        @keyframes codlab-spin { to { transform: rotate(360deg); } }
        
        @media (max-width: 520px) {
            #codlab-popup {
                left: 10px !important;
                right: 10px !important;
                bottom: 75px !important;
                top: auto !important;
                width: auto !important;
                height: calc(100vh - 90px) !important;
                max-height: calc(100vh - 90px) !important;
                border-radius: 12px;
            }
            #codlab-chat-btn .btn-text { display: none; }
            #codlab-chat-btn { padding: 14px; border-radius: 50%; }
        }
        </style>
        <?php
    }

    private function render_html() {
        $url = esc_url($this->options['booking_url']);
        $btn_text = esc_html($this->get_text('button_text'));
        $header = esc_html($this->get_text('header_title'));
        $close_label = esc_attr($this->get_text('close'));
        $show_header = $this->options['show_header'] === '1';
        ?>
        <div id="codlab-booking-widget" data-lang="<?php echo esc_attr($this->current_lang); ?>">
            <button id="codlab-chat-btn" aria-label="<?php echo $btn_text; ?>">
                <svg class="icon-open" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <rect x="3" y="4" width="18" height="18" rx="2"></rect>
                    <line x1="16" y1="2" x2="16" y2="6"></line>
                    <line x1="8" y1="2" x2="8" y2="6"></line>
                    <line x1="3" y1="10" x2="21" y2="10"></line>
                </svg>
                <svg class="icon-close" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <line x1="18" y1="6" x2="6" y2="18"></line>
                    <line x1="6" y1="6" x2="18" y2="18"></line>
                </svg>
                <span class="btn-text"><?php echo $btn_text; ?></span>
            </button>
            
            <div id="codlab-popup">
                <?php if ($show_header): ?>
                <div class="codlab-header">
                    <span class="codlab-header-title"><?php echo $header; ?></span>
                    <button class="codlab-close" aria-label="<?php echo $close_label; ?>">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <line x1="18" y1="6" x2="6" y2="18"></line>
                            <line x1="6" y1="6" x2="18" y2="18"></line>
                        </svg>
                    </button>
                </div>
                <?php endif; ?>
                <div class="codlab-body loading">
                    <iframe id="codlab-iframe" data-src="<?php echo $url; ?>" title="<?php echo $header; ?>"></iframe>
                </div>
            </div>
        </div>
        <?php
    }

    private function render_scripts() {
        ?>
        <script id="codlab-booking-script">
        (function(){
            var btn = document.getElementById('codlab-chat-btn');
            var popup = document.getElementById('codlab-popup');
            var closeBtn = popup.querySelector('.codlab-close');
            var iframe = document.getElementById('codlab-iframe');
            var body = popup.querySelector('.codlab-body');
            var loaded = false;

            btn.onclick = function(e) {
                e.preventDefault();
                popup.classList.contains('open') ? close() : open();
            };

            if (closeBtn) closeBtn.onclick = function(e) { e.preventDefault(); close(); };

            document.onkeydown = function(e) {
                if (e.key === 'Escape' && popup.classList.contains('open')) close();
            };

            document.addEventListener('click', function(e) {
                if (popup.classList.contains('open') && !popup.contains(e.target) && !btn.contains(e.target)) close();
            });

            function open() {
                if (!loaded) {
                    iframe.src = iframe.dataset.src;
                    iframe.onload = function() { body.classList.remove('loading'); };
                    loaded = true;
                }
                popup.classList.add('open');
                btn.classList.add('active');
            }

            function close() {
                popup.classList.remove('open');
                btn.classList.remove('active');
            }
        })();
        </script>
        <?php
    }

    private function get_position_css($pos) {
        $positions = array(
            'bottom-left' => 'bottom: 20px; left: 20px;',
            'top-right' => 'top: 20px; right: 20px;',
            'top-left' => 'top: 20px; left: 20px;',
        );
        return $positions[$pos] ?? 'bottom: 20px; right: 20px;';
    }

    private function get_popup_position_css($pos) {
        $positions = array(
            'bottom-left' => 'bottom: 85px; left: 20px;',
            'top-right' => 'top: 85px; right: 20px;',
            'top-left' => 'top: 85px; left: 20px;',
        );
        return $positions[$pos] ?? 'bottom: 85px; right: 20px;';
    }

    public function add_admin_menu() {
        add_menu_page('CODLAB Booking', 'Booking Widget', 'manage_options', 'codlab-booking-widget', array($this, 'render_admin_page'), 'dashicons-calendar-alt', 80);
    }

    public function register_settings() {
        register_setting('codlab_booking_options', 'codlab_booking_options', array($this, 'sanitize_options'));
    }

    public function sanitize_options($input) {
        return array(
            'booking_url' => esc_url_raw($input['booking_url'] ?? ''),
            'button_text' => sanitize_text_field($input['button_text'] ?? ''),
            'header_title' => sanitize_text_field($input['header_title'] ?? ''),
            'primary_color' => sanitize_hex_color($input['primary_color'] ?? '#F58220'),
            'text_color' => sanitize_hex_color($input['text_color'] ?? '#ffffff'),
            'position' => in_array($input['position'], array('bottom-right', 'bottom-left', 'top-right', 'top-left')) ? $input['position'] : 'bottom-right',
            'button_style' => in_array($input['button_style'], array('rounded', 'square', 'circle')) ? $input['button_style'] : 'rounded',
            'show_on_mobile' => isset($input['show_on_mobile']) ? '1' : '0',
            'popup_width' => max(350, min(600, intval($input['popup_width'] ?? 420))),
            'popup_height' => max(500, min(900, intval($input['popup_height'] ?? 700))),
            'z_index' => max(1, intval($input['z_index'] ?? 999999)),
            'enabled' => isset($input['enabled']) ? '1' : '0',
            'auto_language' => isset($input['auto_language']) ? '1' : '0',
            'show_header' => isset($input['show_header']) ? '1' : '0',
        );
    }

    public function admin_scripts($hook) {
        if ($hook !== 'toplevel_page_codlab-booking-widget') return;
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
    }

    private function get_supported_languages() {
        return array('en'=>'English','de'=>'Deutsch','it'=>'Italiano','fr'=>'Français','es'=>'Español','pt'=>'Português','nl'=>'Nederlands','pl'=>'Polski','ru'=>'Русский','zh'=>'中文','ja'=>'日本語','ko'=>'한국어','tr'=>'Türkçe','ar'=>'العربية','cs'=>'Čeština','da'=>'Dansk','sv'=>'Svenska','no'=>'Norsk','fi'=>'Suomi','el'=>'Ελληνικά','hu'=>'Magyar','ro'=>'Română','uk'=>'Українська','hr'=>'Hrvatski','sk'=>'Slovenčina','sl'=>'Slovenščina');
    }

    public function render_admin_page() {
        $opts = $this->options;
        $detected_lang = $this->current_lang;
        $lang_name = $this->get_supported_languages()[$detected_lang] ?? $detected_lang;
        ?>
        <div class="wrap">
            <h1 style="display:flex;align-items:center;gap:10px;">
                <span style="background:#F58220;color:#fff;padding:8px 12px;border-radius:8px;font-weight:bold;">
                    CO<span style="border:2px solid #fff;border-radius:4px;padding:0 4px;">D</span>LAB
                </span>
                Booking Widget <small style="color:#666;font-weight:normal;">v2.2</small>
            </h1>
            
            <?php if (empty($opts['booking_url'])): ?>
            <div class="notice notice-warning"><p><strong>⚠️ Configura l'URL di prenotazione</strong> per attivare il widget.</p></div>
            <?php endif; ?>

            <div class="notice notice-info" style="background:#f0f6fc;border-left-color:#2271b1;">
                <p><strong>🌍 Lingua rilevata:</strong> <?php echo esc_html($lang_name); ?> (<code><?php echo esc_html($detected_lang); ?></code>)</p>
            </div>

            <form method="post" action="options.php">
                <?php settings_fields('codlab_booking_options'); ?>
                
                <div style="display:grid;grid-template-columns:1fr 1fr;gap:30px;margin-top:20px;">
                    <div>
                        <div class="postbox" style="padding:20px;">
                            <h2 style="margin-top:0;">⚙️ Impostazioni Principali</h2>
                            <table class="form-table">
                                <tr>
                                    <th>Widget Attivo</th>
                                    <td><label><input type="checkbox" name="codlab_booking_options[enabled]" value="1" <?php checked($opts['enabled'], '1'); ?>> Mostra il widget sul sito</label></td>
                                </tr>
                                <tr>
                                    <th>URL Prenotazione *</th>
                                    <td><input type="url" name="codlab_booking_options[booking_url]" value="<?php echo esc_attr($opts['booking_url']); ?>" class="large-text" placeholder="https://booking.codlab.de/prenota/..."></td>
                                </tr>
                            </table>
                        </div>

                        <div class="postbox" style="padding:20px;">
                            <h2 style="margin-top:0;">🌍 Lingua e Testi</h2>
                            <table class="form-table">
                                <tr>
                                    <th>Lingua Automatica</th>
                                    <td><label><input type="checkbox" name="codlab_booking_options[auto_language]" value="1" <?php checked($opts['auto_language'], '1'); ?>> Traduci automaticamente</label></td>
                                </tr>
                                <tr>
                                    <th>Testo Bottone</th>
                                    <td><input type="text" name="codlab_booking_options[button_text]" value="<?php echo esc_attr($opts['button_text']); ?>" class="regular-text" placeholder="<?php echo esc_attr($this->translations['button_text']); ?>"></td>
                                </tr>
                                <tr>
                                    <th>Mostra Header</th>
                                    <td><label><input type="checkbox" name="codlab_booking_options[show_header]" value="1" <?php checked($opts['show_header'], '1'); ?>> Mostra header popup (disattiva se la pagina ha già un header)</label></td>
                                </tr>
                                <tr>
                                    <th>Titolo Header</th>
                                    <td><input type="text" name="codlab_booking_options[header_title]" value="<?php echo esc_attr($opts['header_title']); ?>" class="regular-text" placeholder="<?php echo esc_attr($this->translations['header_title']); ?>"></td>
                                </tr>
                            </table>
                        </div>

                        <div class="postbox" style="padding:20px;">
                            <h2 style="margin-top:0;">🎨 Colori</h2>
                            <table class="form-table">
                                <tr>
                                    <th>Colore Primario</th>
                                    <td><input type="text" name="codlab_booking_options[primary_color]" value="<?php echo esc_attr($opts['primary_color']); ?>" class="color-picker"></td>
                                </tr>
                                <tr>
                                    <th>Colore Testo</th>
                                    <td><input type="text" name="codlab_booking_options[text_color]" value="<?php echo esc_attr($opts['text_color']); ?>" class="color-picker"></td>
                                </tr>
                            </table>
                        </div>
                    </div>
                    
                    <div>
                        <div class="postbox" style="padding:20px;">
                            <h2 style="margin-top:0;">📐 Layout</h2>
                            <table class="form-table">
                                <tr>
                                    <th>Posizione</th>
                                    <td>
                                        <select name="codlab_booking_options[position]">
                                            <option value="bottom-right" <?php selected($opts['position'], 'bottom-right'); ?>>↘️ Basso Destra</option>
                                            <option value="bottom-left" <?php selected($opts['position'], 'bottom-left'); ?>>↙️ Basso Sinistra</option>
                                            <option value="top-right" <?php selected($opts['position'], 'top-right'); ?>>↗️ Alto Destra</option>
                                            <option value="top-left" <?php selected($opts['position'], 'top-left'); ?>>↖️ Alto Sinistra</option>
                                        </select>
                                    </td>
                                </tr>
                                <tr>
                                    <th>Stile Bottone</th>
                                    <td>
                                        <select name="codlab_booking_options[button_style]">
                                            <option value="rounded" <?php selected($opts['button_style'], 'rounded'); ?>>💊 Arrotondato</option>
                                            <option value="square" <?php selected($opts['button_style'], 'square'); ?>>⬜ Quadrato</option>
                                            <option value="circle" <?php selected($opts['button_style'], 'circle'); ?>>⚫ Cerchio</option>
                                        </select>
                                    </td>
                                </tr>
                                <tr>
                                    <th>Dimensioni Popup</th>
                                    <td>
                                        <input type="number" name="codlab_booking_options[popup_width]" value="<?php echo esc_attr($opts['popup_width']); ?>" min="350" max="600" style="width:70px;"> x 
                                        <input type="number" name="codlab_booking_options[popup_height]" value="<?php echo esc_attr($opts['popup_height']); ?>" min="500" max="900" style="width:70px;"> px
                                    </td>
                                </tr>
                                <tr>
                                    <th>Mobile</th>
                                    <td><label><input type="checkbox" name="codlab_booking_options[show_on_mobile]" value="1" <?php checked($opts['show_on_mobile'], '1'); ?>> Mostra su dispositivi mobili</label></td>
                                </tr>
                                <tr>
                                    <th>Z-Index</th>
                                    <td><input type="number" name="codlab_booking_options[z_index]" value="<?php echo esc_attr($opts['z_index']); ?>" min="1" style="width:100px;"></td>
                                </tr>
                            </table>
                        </div>

                        <div class="postbox" style="padding:15px;background:#e8f5e9;">
                            <h3 style="margin-top:0;">🌐 26 Lingue Supportate</h3>
                            <div style="display:flex;flex-wrap:wrap;gap:4px;">
                                <?php foreach ($this->get_supported_languages() as $code => $name): ?>
                                <span style="background:<?php echo $code === $detected_lang ? '#F58220' : '#fff'; ?>;color:<?php echo $code === $detected_lang ? '#fff' : '#333'; ?>;padding:2px 6px;border-radius:3px;font-size:11px;border:1px solid #ddd;" title="<?php echo esc_attr($name); ?>"><?php echo $code; ?></span>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </div>

                <?php submit_button('💾 Salva Impostazioni', 'primary large'); ?>
            </form>
        </div>
        <script>jQuery(document).ready(function($) { $('.color-picker').wpColorPicker(); });</script>
        <?php
    }
}

CODLAB_Booking_Widget::get_instance();
